import numpy as np
def basic_int_analysis(data,time,t_arg):
    """
    This performs basic statistical analysis on the data set broken into windows of time.
    
        Input
        -----
        data : numpy array
            data you would like statistical evaluation of.
        time : numpy array
            time array associated with data
        t_arg : tuple
            (window,unit) - 
            window : the time window you would like the data to be broken up into.
            unit : the unit of time to consider with the window variable, valid units are
            's' (seconds),
            'm' (minutes),
            'h' (hours),
            'd' (days),
            'w' (weeks).
        
        Output
        ------
        int_data : list of tuples
            int_data stores the following per tuple:
                1. The time data of the interval in seconds (the time data is converted back before it is returned). type: numpy array
                2. The data associated with the time data. type: numpy array
                3. The basic statistics of the associated interval data.
            The data is stored for external plotting and processing purposes. 
    """
    # Check that t_arg is of correct format
    valid = ['s','m','h','d','w']
    if not t_arg[1] in valid:
        raise TypeError("Time unit '"+t_arg[1]+ "' not supported. \n Valid time units are: \n 's' (seconds),'m' (minutes),'h' (hours),'d' (days), and 'w' (weeks).")
    
    # We know the timestamp is in seconds, so we assign conversion unit accordingly.
    
    if t_arg[1] == 'm':
        conv = 60
    elif t_arg[1] == 'h':
        conv = 60*60
    elif t_arg[1] == 'd':
        conv = 60*60*24
    elif t_arg[1] == 'w':
        conv = 60*60*24*7
    else: 
        conv = 1
        
    # Divide time vector into desired units.
    re_t = time/conv
    
    # Determine if there is at least one of the desired time windows in the timestream, otherwise return an error
    t_time = re_t[-1]-re_t[0]
    
    if t_time < t_arg[0]:
        raise Exception("The time window is greater than the total time vector. \n Time window needs to be a subset of the total time vector.")
    
    # Break time series into time window and calculate associated statistics
    intervals = int(np.floor(t_time/t_arg[0]))
    int_data = []
   	
    for i in range(intervals):
        interval_ind = np.where((re_t < re_t[0]+(i+1)*t_arg[0]) & (re_t >= re_t[0]+i*t_arg[0]))
        t_int = re_t[interval_ind]
        data_int = data[interval_ind]
        stat = [np.amin(data_int),np.amax(data_int),data_int[np.where((data_int <= .09) & (data_int >= -.09))],np.mean(data_int),np.std(data_int)]
        
        int_data.append((t_int*conv,data_int,stat))
        
    return int_data