# -*- coding: utf-8 -*-
"""
A script for making Multi-PIP mode files
Created on Tue Mar 23 09:58:35 2021

This script file is designed for Multi PIP data files versions 2.00 and newer. 
This script generates a mode file by considering 3 running average windows of the raw data for different sets 
of frequency values. The first range uses raw data for frequencies below 500 Hz. The second is a 7 point running mean 
for 500 - 5000 Hz. The third is 50 point running mean for 500 - 100500 Hz. The script will plot each of the regions
with the fits for visualization. 
"""

import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import matplotlib as mpl
from matplotlib.backends.backend_pdf import PdfPages
import csv

# Data file to use for mode calculation (NEEDS MODIFICATION)
d_file = 'C:\\example\\path\\example_mode_data.CSV'


# Drive voltage
V = 2

# Value of resistor (in ohms)
R = 10000000

# Define output file name (NEEDS MODIFICATION)
out_file = 'C:\\example\\out\\path\\test_'+str(int(R/1000))+'K_V_'+str(V)+'_GAIN_MODE.CSV'

print("Making Mode...")
# Make header (CHECK THIS AND MODIFY MANUALLY)
d = {'Property' : ['"Load Resistor"', '"Input Gain"', '"Output Vpp"', '"Rfb Select"'],
     'Value': [str(R),'IA_CTRL_PGA_GAIN1', 'IA_CTRL_VOLT_2000', 'false' ] }

top = pd.DataFrame({'1': ['Multi-PIP Response Magnitude to Impedance modeibration']})
name = pd.DataFrame({'1':['Name'],'2': ['High 10M Mode']})
head = pd.DataFrame(data = d)

# Import data
d_mag = pd.read_csv(d_file, skiprows = 3)
#print(d_mag.iloc[:,7])

# Compute moving average for two different windows and plot them all
d_mag['Moving average 7 points'] = d_mag.iloc[:,7].rolling(window=7).mean()
d_mag['Moving average 50 points'] = d_mag.iloc[:,7].rolling(window=50).mean()
fig, ax = plt.subplots(1,figsize = (12,8))
ax.semilogx(d_mag['frequency'],d_mag['magnitude'], c = 'b', label = 'Raw')
ax.semilogx(d_mag['frequency'],d_mag['Moving average 7 points'], c = 'r', label = '7 point window mean')
ax.semilogx(d_mag['frequency'],d_mag['Moving average 50 points'], c = 'g', label = '50 point window mean')
ax.set_xlabel('Frequency (Hz)')
ax.set_ylabel('Magnitude (Counts)')
#ax.set_ylim([0,500])
fig.suptitle('Different running mean windows for '+str(int(R/1000))+' K Mode')
ax.legend()


# Set up mode file data
mode = pd.DataFrame(columns = ['"Frequency (Hz)"', '"Gain"'])
mid_mode = pd.DataFrame()
high_mode = pd.DataFrame()

# Find the three windows of gain values
idx_low = d_mag.loc[(d_mag['frequency'] < 500)]
idx_mid = d_mag.loc[(d_mag['frequency'] >= 500) & (d_mag['frequency'] < 5000)]
idx_high = d_mag.loc[(d_mag['frequency'] >= 5000) & (d_mag['frequency'] < 100500)]

# For low range profile range, the gain is just computed from the raw data
mode['"Frequency (Hz)"'] = idx_low['frequency']
mode['"Gain"'] = 1/(R*idx_low['magnitude'])


# For mid frequency range, the gain is computed from 7 point moving mean
mid_mode['"Frequency (Hz)"'] = idx_mid['frequency']
mid_mode['"Gain"'] = 1/(R*idx_mid['Moving average 7 points'])

mode = mode.append(mid_mode)  

# For high frequency range, the gain is computed from 50 point moving mean
high_mode['"Frequency (Hz)"'] = idx_high['frequency']
high_mode['"Gain"'] = 1/(R*idx_high['Moving average 7 points'])

mode = mode.append(high_mode) 

# Open and erase data already present in file
f = open(out_file,'w')
f.truncate()
f.close()
        
f = open(out_file,'a',newline='')
        
# Write metadata and data to csv file.
top.to_csv(f, line_terminator="", index = False, header = False, quoting = 1)
name.to_csv(f, line_terminator="", index = False, header = False, quoting = 1)
head.to_csv(f, line_terminator="", index = False, header = False, quoting = csv.QUOTE_NONE)

# Write mode data to CSV
mode.to_csv(f, line_terminator="", index = False, quoting = csv.QUOTE_NONE)
        
f.close()

print("Done!")
