import pandas as pd
import numpy as np 
from datetime import datetime, timedelta
import matplotlib as mpl
import matplotlib.pyplot as plt

class MP:
    """
    The Multi-PIP class.
    First Authored: 2/28/2021 
    
    The methods contained in MP are grouped by functionality and outlined with brief descriptions below.
    
    Data I-O and Filtering Related Methods
    ------------------------
        load : Imports data from CSV.
        save_NDZI : Save NDZI data as a CSV.
        save_Phase : Save phase angle data as a CSV.
        save_Real_Imaginary : Save real and imaginary data as a CSV.
        save_Impedance : Save impedance data as a CSV.
        save_Temperature : Save temperature data as a CSV.
        save_Magnitude : Save Magnitude data as a CSV.
        z_window : An impedance filter that discards all data outside a user-defined window.
    
    Calculation Related Methods
    ---------------------------
        calc_NDZI : Calculates NDZI for all probe pairs.
    
    Plotting Related Methods
    ------------------------
        plot_NDZI : Plots NDZI for one probe pair.
        plot_z_vs_t : Plots impedance versus time for one probe pair.
        plot_z_h_vs_z_l : Plot high impedance versus low impedance for one probe pair.
        plot_T_vs_t : plot |Phase angle| versus time for a particular frequency and probe pair.
        plot_z_ratio_vs_time : Plot ratio of impedance (low frequency) / impedance (high frequency).
        plot_R_vs_I : Plot real versus imaginary for all frequencies for a particular probe pair.
        plot_z_h_vs_T_h : Plot impedance versus |phase angle| for a particular frequency.
        
    Formatting and Checking Related Methods
    ---------------------------------------
        format_datetime_plot : Sets title and axis format for datetime plot.
        prb_pair_name_crosscheck : Crosschecks probe pair with probe pair name.
        
    
        
    Please see method descriptions for more detailed documentation on each of the methods.

    """
    version = None
    name = None
    metadata = pd.DataFrame()
    data = pd.DataFrame()
    NDZI = pd.DataFrame()
    
    def __init__(self):
        print("You have initialized the Multi-PIP data handler. \n Please upload a CSV data file!\n\n")

    def load(self,path):
        """
            load imports data from a CSV file and saves it to the data and metadata attributes of MP.
            If something goes wrong loading the file it throws an exception. load will load data differently
            according to different Multi-PIP versions it finds in the header of the CSV file.
            
            Inputs
            ------
            path : string. 
                The path to the CSV data file. 
            
        """
        try:
            metadata_init = pd.read_csv(path,header=None,usecols=[1,2], nrows=4).transpose()
            metadata_init.columns = ['Device Info','Start Time','End Time','Gain Setting']
            self.version = metadata_init.iloc[1]['Device Info']
            
            # Based on version, import your data.
            if float(self.version) < 2.02:
                try:
                    # Load metadata
                    self.metadata = pd.read_csv(path,header=None,usecols=[1,2], nrows=4).transpose()
                    self.metadata.columns = ['Device Info','Start Time','End Time','Gain Setting']
                    self.name = self.metadata.iloc[0]['Device Info']
                    self.calibration = self.metadata.iloc[0]['Gain Setting']
                    
                    # Load data
                    self.data = pd.read_csv(path,skiprows=4)
                    print('WARNING: \nYou are using Multi-PIP software version '+self.version+', which is no longer supported.\nYou may attempt to use the MP analysis suite, though anything to do with the "Mode" column will result in errors.\n\n')
                except:
                    print("Something went wrong loading your data file. Check your path and data file format and try again.\n\n")
                    
            
            else:
                try:
                    # Load metadata
                    self.metadata = pd.read_csv(path,header=None,usecols=[1,2], nrows=3).transpose()
                    self.metadata.columns = ['Device Info','Start Time','End Time']
                    self.name = self.metadata.iloc[0]['Device Info']
                    
                    # Load data
                    self.data = pd.read_csv(path,skiprows=3)
                    print('NOTE: \nYou are using Multi-PIP software version '+self.version+'.\nAll mode data is stored in a separate column titled "Mode".\n\n')
                except:
                    print("Something went wrong loading your data file. Check your path and data file format and try again.\n\n")
            
            print("Loaded data from device "+self.name+". \nThe data file loaded has "+str(self.data.shape)+ " rows and columns.\n\n")
            
        except:
            print("Something went wrong loading your data file. Check your path and data file format and try again.\n\n")
        

    def calc_NDZI(self, f_l, f_h, df = pd.DataFrame()):
        """
            calc_NDZI computes the NDZI associated with specific user entered frequencies. 
            It then saves the current NDZI under the self.NDZI attribute.
            
            Inputs
            ------
            f_l : (required) integer. 
                Low frequency to use for NDZI computation.
            f_h : (required) integer. 
                High frequency to use for NDZI computation.
            df : (optional) pandas dataframe.
                The dataframe from which to calculate NDZI. Default is self.data
            
            Outputs
            -------
            NDZI : pandas dataframe. 
                Contains NDZI values associated with f_l and f_h as well as the associated probe pair name and timestamp for each of the values.
        
        """
        print("Calculating NDZI for "+str(f_l)+" and "+str(f_h)+"...")
        
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
               
        # Find probe pair names and frequency vector
        freq = pd.unique(df['frequency'])
        prb_pair_names = pd.unique(df['probe pair name'])
        prb_pairs = pd.unique(df['probe pair'])
        # Check if frequencies inputted are in vector
        if not (f_l in freq) & (f_h in freq):
            raise Exception("One of the frequencies you entered: ("+str(f_l)+","+str(f_h)+") is not contained in the data.")
            
        
        # Initialize NDZI dataframe
        NDZI = pd.DataFrame(columns=["Time","probe pair","probe pair name","NDZI","High freq","Low freq"])
        for cnt,i in enumerate(prb_pair_names):

            # Find indices associated with high and low frequency and probe pair
            ind_f_l = df.loc[(df['frequency'] == f_l) & (df['probe pair name'] == i)]
            ind_f_h = df.loc[(df['frequency'] == f_h) & (df['probe pair name'] == i)]
            
            # Debugging check
            #print(len(ind_f_l), len(ind_f_h))
            
            # Get impedance and time
            ind_z_l = ind_f_l['impedance']/1000
            ind_z_h = ind_f_h['impedance']/1000
            t = np.array(ind_f_l['Time'])
                    
            # Calculate NDZI
            ind_NDZI = np.divide(np.array(ind_z_l) - np.array(ind_z_h) , np.array(ind_z_l) + np.array(ind_z_h))*100
                    
            # Save all values to the NDZI dataframe
            for l in range(len(t)):
                NDZI = NDZI.append({"Time" : t[l], "probe pair" : prb_pairs[cnt], "probe pair name" : i, "NDZI" : ind_NDZI[l], "High freq" : f_h, "Low freq" : f_l},  ignore_index = True)
        
        # Save NDZI to attribute
        self.NDZI = NDZI
        print(" Done!\n\n")
    
        return NDZI
    
    def plot_NDZI(self, prb_pair = None, prb_pair_name = None, f_l = None, f_h = None, df = pd.DataFrame()):
        """
            plot_NDZI takes care of all of the formatting for creating a datetime/NDZI plot. 
            
            Note: If you inputted a custom dataframe (i.e., input df is nonempty) it should be the data FROM WHICH 
            to calculate NDZI, and not precalculated NDZI data. 
            For example:
                (Correct Usage)
                    filtered_data = PIP.z_window(1,1000)
                    plot_NDZI(df = filtered_data)
                (Incorrect Usage)
                    filtered_data = PIP.z_window(1,1000)
                    filtered_NDZI = PIP.calc_NDZI(100, 95000, df = filtered_data)
                    plot_NDZI(df = filtered_NDZI)
            
            Inputs
            ------
            (ONE OF THE FOLLOWING IS REQUIRED)
            prb_pair: string. 
                The probe pair of the data to plot.  
            prb_pair_name : string. 
                The probe pair name of the data to plot. 
            
            df : (optional) pandas dataframe. 
                Contains data from which to calculate and plot NDZI. Default is to go get NDZI data from self.NDZI.
            f_l : (optional) number.
                Low frequency to plot NDZI data for. Default is min frequency in self.data.
            f_h : (optional) number.
                High frequency to plot NDZI data for. Default is max frequency in self.data.
                
            Outputs
            -------
            fig : matplotlib figure.
                Figure associated with the NDZI plot.
            ax : matplotlib axes.
                Axis associated with the NDZI plot.
            
        """
        
        
        if df.empty:
            
            if (f_l != None) and (f_h != None):
                freq = pd.unique(self.data['frequency'])
                # Check if high and low frequencies are in data.
                if not(f_l in freq):
                    raise TypeError('Custom low frequency not found in self.data.')
                elif not (f_h in freq):
                    raise TypeError('Custom high frequency not found in self.data.')
                else:
                    # If no NDZI is inputted, calculate NDZI with highest and lowest frequency found in the data.
                    df = self.calc_NDZI(f_l,f_h)
                
            else:
                freq = pd.unique(self.data['frequency'])
                f_l = freq[0] 
                f_h = freq[-1]
                print("NDZI not computed yet.\n Using default frequency values "+str(f_l)+" and "+str(f_h)+" Hz to compute NDZI. \n")
                df = self.calc_NDZI(f_l,f_h)
        else:
            # If personal dataframe is passed in, force NDZI calculation
            if f_l == None or f_h == None:
                freq = pd.unique(df['frequency'])
                f_l = freq[0]
                f_h = freq[-1]
            print("Passed in special dataframe, calculating NDZI for this data.")
            df = self.calc_NDZI(f_l, f_h, df = df) 
            
                
        # Find frequencies.
        f_h = pd.unique(df['High freq'])
        f_l = pd.unique(df['Low freq'])
        
        #Check wether inputs correspond to good data
        prb_pair, prb_pair_name = self.prb_pair_name_crosscheck(df, prb_pair, prb_pair_name)
        
        print("Plotting NDZI for probe pair name "+prb_pair_name+" and associated probe pair "+prb_pair+"...")
        
        # Get NDZI and time data for particular probe pair
        ind_prb_pr = df.loc[df['probe pair'] == prb_pair]
            
        NDZI = ind_prb_pr['NDZI']
        t = ind_prb_pr['Time']
        t_i = pd.to_datetime(t.min(),unit='s')
        t_f = pd.to_datetime(t.max(),unit='s')
            
        # Get datetime title and axis format
        title_form, ax_form = self.format_datetime_plot(t_i,t_f)
            
        
        # Plot NDZI 
        fig, ax = plt.subplots(1, figsize = (12,8))
        fig.suptitle("NDZI ("+str(f_l[0])+","+str(f_h[0])+" Hz), Probe: "+prb_pair+", Name: "+prb_pair_name+", "+title_form)
        ax.scatter(pd.to_datetime(t,unit='s'),NDZI)
        ax.set_ylim([0,100])
        ax.set_ylabel('NDZI')   
        ax.xaxis.set_major_formatter(mpl.dates.DateFormatter(ax_form))
        ax.xaxis.set_minor_formatter(mpl.dates.DateFormatter(ax_form))
    
        ax.legend([prb_pair_name])
        
        print("Done!\n")
        return fig, ax
    
    def format_datetime_plot(self,t_i,t_f):
        """
            format_datetime_plot sets axis datetime format options and title date information based on elapsed time for windows. 
    
            Input
            -----
            t_i : pandas datetime
                Initial time
            t_f : pandas datetime
                Final Time

            Output
            ------
            ax_form: string
                Axis datetime format
            title_form: string
                Date data to include in title e.g. 2021/05/07 -----> 2021/08/19
                
        """
        # Calculate elapsed time between initial and final
        delta_t = t_f - t_i 
        
        # Determine title and axis format based off elapsed time
        if delta_t <= timedelta(seconds = 300):
            ax_form = "%M:%S"
            title_form = t_f.strftime("%Y/%m/%d")+", hour "+t_f.strftime("%H")
        elif (delta_t > timedelta(seconds = 300))&(delta_t <= timedelta(minutes = 80)):
            ax_form = "%H:%M"
            title_form = t_i.strftime("%Y/%m/%d")+" -----> "+t_f.strftime("%Y/%m/%d")
        elif (delta_t > timedelta(minutes = 80))&(delta_t <= timedelta(hours = 72)):
            ax_form = "%H:%M"
            title_form = t_i.strftime("%Y/%m/%d")+" -----> "+t_f.strftime("%Y/%m/%d")
        elif (delta_t > timedelta(hours = 72))&(delta_t <= timedelta(days = 14)):
            ax_form = "%m/%d"
            title_form = t_i.strftime("%Y/%m/%d")+" -----> "+t_f.strftime("%Y/%m/%d")
        else: 
            ax_form = "%m/%d"
            title_form = t_i.strftime("%Y/%m/%d")+" -----> "+t_f.strftime("%Y/%m/%d")
        
        
        return title_form,ax_form
    
    def prb_pair_name_crosscheck(self,df,prb_pair,prb_pair_name):
        """
            prb_pair_name_crosscheck validates that the inputted probe pair and probe pair name correspond to the same data.
            prb_pair_name_crosscheck checks wether or not the probe pair and name are contained in the dataframe.
            Then it does the following based on the cases:
                1. Neither are contained:
                    Throws exception saying it cannot find the inputted probe pair and name.
                2. One of the two is contained in the dataframe
                    Finds the corresponding probe pair or name and returns them. 
                3. Both are contained
                    Performs crosscheck to make sure there is a one to one correspondence between probe pair name and probe pair.
                    Otherwise it throws an exception. 
                    
    
            Input
            -----
            df : pandas dataframe
                Data to search for probe pair and probe pair name
            prb_pair : string
                Probe pair to crosscheck with probe pair name, e.g., 'A1C1'.
            prb_pair_name : string
                Probe pair name to crosscheck with probe pair, e.g., 'TYPE1_1_LEAF_1'.

            Output
            ------
            prb_pair: string
                Probe pair corresponding to probe pair name. 
            prb_pair_name: string
                Probe pair name corresponding to probe pair.
                
        """
        
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
            
        prb_pairs = pd.unique(df['probe pair'])
        prb_pair_names = pd.unique(df['probe pair name'])
        
        if (not prb_pair in prb_pairs) and (not prb_pair_name in prb_pair_names):
            raise Exception("Neither the probe pair nor the probe pair name was found in the data set.\n")
            
        elif (prb_pair in prb_pairs) and (not prb_pair_name in prb_pair_names):
            ind_prb = df.loc[df['probe pair'] == prb_pair]
            prb_pair_name = pd.unique(ind_prb['probe pair name'])[0]
            
            
        elif (not prb_pair in prb_pairs) and (prb_pair_name in prb_pair_names):
            
            ind_prb_pair_names = df.loc[df['probe pair name'] == prb_pair_name]
            prb_pair = pd.unique(ind_prb_pair_names['probe pair'])[0]
            
        else:
            ind_prb = df.loc[df['probe pair'] == prb_pair]
            ind_prb_name = df.loc[df['probe pair name'] == prb_pair_name]
            
            pair_to_name = pd.unique(ind_prb['probe pair name'])
            name_to_pair = pd.unique(ind_prb_name['probe pair'])
            
            if len(pair_to_name) > 1:
                raise Exception("The entered probe pair corresponds to more than one probe pair name.\n")
                
            elif len(name_to_pair) > 1:
                raise Exception("The entered probe pair name corresponds to more than one probe pair.\n")
                
            if pair_to_name[0] != prb_pair_name or name_to_pair[0] != prb_pair:
                raise Exception("Your input probe pair and probe pair name do not correspond to the same data set.\n")
                
        return prb_pair, prb_pair_name
    
    def plot_z_vs_t(self, freq, df = pd.DataFrame(), prb_pair = None, prb_pair_name = None, z_upp_bnd = None, z_low_bnd = None):
        """
            plot_z_vs_t takes care of all of the formatting for creating a datetime/Impedance (Z) plot. 
            
            Inputs
            ------
            freq : (required) integer.
                The frequency for which impedance is to be plotted. 
                
            (ONE OF THE FOLLOWING IS REQUIRED)
            prb_pair: string. 
                The probe pair of the data to plot.  
            prb_pair_name : string. 
                The probe pair name of the data to plot. 
                
            df : (optional) pandas dataframe. 
                Data to be plotted. Should have the same format as all other Multi-PIP data. Defaults to self.data.
            z_upp_bnd : (optional) float.
                The upper bound for the impedance vs time plot in kOhms. Default is 'z_upp_bnd = Z.max() + 10 (kOhms)'.
            z_low_bnd : (optional) float.
                The lower bound for the impedance vs time plot in kOhms. Default is 'z_low_bnd = 0'.
                
            Outputs
            -------
            fig : matplotlib figure.
                Figure associated with the impedance versus time plot.
            ax : matplotlib axes.
                Axis associated with the impedance versus time plot.
            
        """
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
        
        #Check that given frequency is part of the data set
        frequencies = pd.unique(df['frequency'])
        if not freq in frequencies:
            raise TypeError("The frequency you inputted does not match any of the frequencies in the data.")
        
        #Check wether inputs correspond to good data
        prb_pair, prb_pair_name = self.prb_pair_name_crosscheck(df, prb_pair, prb_pair_name)
        
        print("Plotting Impedance versus time at "+str(freq)+" Hz for probe pair name "+prb_pair_name+" and associated probe pair "+prb_pair+"...")
            
    
        # Find indices associated with frequency and probe pair
        ind = df.loc[(df['frequency'] == freq) & (df['probe pair name'] == prb_pair_name)]
        
                    
        # Get impedance and time
        Z = ind['impedance']/1000
        t = ind['Time']
        
        # Set upper and lower axis bounds for impedance
        if z_upp_bnd == None:
            z_upp_bnd = Z.max()+10
        
        if z_low_bnd == None:
            z_low_bnd = 0
            
        # Get initial and final time
        t_i = pd.to_datetime(t.min(),unit='s')
        t_f = pd.to_datetime(t.max(),unit='s')
        
        # Get datetime title and axis format
        title_form, ax_form = self.format_datetime_plot(t_i,t_f)
        
        # Plot impedance
        fig, ax = plt.subplots(1, figsize = (12,8))
        fig.suptitle("Impedance at "+str(freq)+" Hz, Probe: "+prb_pair+", Name: "+prb_pair_name+", "+title_form)
        ax.scatter(pd.to_datetime(t,unit='s'),Z)
        ax.set_ylim([z_low_bnd,z_upp_bnd])
        ax.set_ylabel('Z (kOhms)')   
        ax.xaxis.set_major_formatter(mpl.dates.DateFormatter(ax_form))
        ax.xaxis.set_minor_formatter(mpl.dates.DateFormatter(ax_form))
    
        ax.legend([prb_pair_name])
        
        print("Done!\n")
        return fig, ax
    
    def plot_z_h_vs_z_l(self, prb_pair = None, prb_pair_name = None, df = pd.DataFrame, f_l = None, f_h = None, z_l_bnd = None, z_h_bnd = None):
        """
            plot_z_h_vs_z_l plots impedance at a high frequency versus impedance at a low frequency for a particular probe pair or name. 
            
            Inputs
            ------
            (ONE OF THE FOLLOWING IS REQUIRED)
            prb_pair: string. 
                The probe pair of the data to plot.  
            prb_pair_name : string. 
                The probe pair name of the data to plot.
            
            df : (optional) pandas dataframe. 
                Data to be plotted. Should have the same format as all other Multi-PIP data. Defaults to self.data.
            f_l : (optional) integer . 
                Low frequency to plot impedance for. Default is minimum frequency found in the data.
            f_h : (optional) integer. 
                High frequency to plot impedance for. Default is maxmimum frequency found in the data.
            z_l_bnd : (optional) float.
                The upper bound for low frequency impdance in kOhms. Default is 'z_l_bnd = Z_l.max() + 10 (kOhms)'.
            z_h_bnd : (optional) float.
                The upper bound for high frequency impedance in kOhms. Default is 'z_l_bnd = Z_h.max() + 10 (kOhms)'.
                
            Outputs
            -------
            fig : matplotlib figure.
                Figure associated with high versus low impedance.
            ax : matplotlib axes.
                Axis associated with high versus low impedance.
                
                
        """
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
        
        # Find frequency vector in data
        freq = pd.unique(df['frequency'])
        # Check if frequencies inputted are in vector
        if not (f_l in freq) & (f_h in freq):
            f_l = freq.min()
            f_h = freq.max()
            print("Could not find frequencies inputted in data, defaulting to :\nf_l = "+str(f_l)+", f_h = "+str(f_h))
            
        
        #Check wether inputs correspond to good data
        prb_pair, prb_pair_name = self.prb_pair_name_crosscheck(df, prb_pair, prb_pair_name)
        
        print("Plotting high vs low impedance at "+str(f_l)+", "+str(f_h)+" Hz for probe pair name "+prb_pair_name+" and associated probe pair "+prb_pair+"...")
        

        # Find indices associated with high and low frequency and probe pair
        ind_f_l = df.loc[(df['frequency'] == f_l) & (df['probe pair name'] == prb_pair_name)]
        ind_f_h = df.loc[(df['frequency'] == f_h) & (df['probe pair name'] == prb_pair_name)]
                    
        # Get impedance and time
        z_l = ind_f_l['impedance']/1000
        z_h = ind_f_h['impedance']/1000
        
        
        # Set axis bounds for high and low impedance
        if z_h_bnd == None:
            z_h_bnd = z_h.max()+50
        
        if z_l_bnd == None:
            z_l_bnd = z_l.max()+50
        
            
        
        # Plot high versus low impedance
        fig, ax = plt.subplots(1, figsize = (12,8))
        fig.suptitle("Low vs high impedance ("+str(f_l)+","+str(f_h)+" Hz), Probe: "+prb_pair+", Name: "+prb_pair_name)
        ax.scatter(z_h,z_l)
        ax.set_xlim([0,z_h_bnd])
        ax.set_ylim([0,z_l_bnd])
        ax.set_xlabel('Z_h (kOhms)')
        ax.set_ylabel('Z_l (kOhms)')
                    
                    
        print(" Done!\n\n")
        return fig, ax
    
    def plot_T_vs_t(self, freq, prb_pair = None, prb_pair_name = None, df = pd.DataFrame(), Theta_upp_bnd = None, Theta_low_bnd = None):
        """
            plot_T_vs_t takes care of all of the formatting for creating a datetime/|Phase Angle| (T) plot. 
            
            Inputs
            ------
            freq : (required) integer.
                The frequency for which impedance is to be plotted. 
                
            (ONE OF THE FOLLOWING IS REQUIRED)
            prb_pair: string. 
                The probe pair of the data to plot.  
            prb_pair_name : string. 
                The probe pair name of the data to plot. 
                
            df : (optional) pandas dataframe. 
                Data to be plotted. Should have the same format as all other Multi-PIP data. Defaults to self.data.
            Theta_upp_bnd : (optional) float.
                The upper bound for the phase angle vs time plot in degrees. Default is 'Theta_upp_bnd = T.max() + 5'.
            Theta_low_bnd : (optional) float.
                The lower bound for the phase angle vs time plot in degrees. Default is 'Theta_low_bnd = 0'.
                
            Outputs
            -------
            fig : matplotlib figure.
                Figure associated with the phase angle versus time plot.
            ax : matplotlib axes.
                Axis associated with the phase angle versus time plot.
            
        """
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
            
        #Check that given frequency is part of the data set
        frequencies = pd.unique(df['frequency'])
        if not freq in frequencies:
            raise TypeError("The frequency you inputted does not match any of the frequencies in the data.")
        
        #Check wether inputs correspond to good data
        prb_pair, prb_pair_name = self.prb_pair_name_crosscheck(df, prb_pair, prb_pair_name)
        
        print("Plotting Impedance versus time at "+str(freq)+" Hz for probe pair name "+prb_pair_name+" and associated probe pair "+prb_pair+"...")
            
    
        # Find indices associated with frequency and probe pair
        ind = df.loc[(df['frequency'] == freq) & (df['probe pair name'] == prb_pair_name)]
        
                    
        # Get |phase| angle and time
        T = ind['phase'].abs()
        t = ind['Time']
        
        # Set upper and lower axis bounds for impedance
        if Theta_upp_bnd == None:
            Theta_upp_bnd = T.max() + 5
        
        if Theta_low_bnd == None:
            Theta_low_bnd = 0
            
        # Get initial and final time
        t_i = pd.to_datetime(t.min(),unit='s')
        t_f = pd.to_datetime(t.max(),unit='s')
        
        # Get datetime title and axis format
        title_form, ax_form = self.format_datetime_plot(t_i,t_f)
        
        # Plot impedance
        fig, ax = plt.subplots(1, figsize = (12,8))
        fig.suptitle("|Phase angle| at "+str(freq)+" Hz, Probe: "+prb_pair+", Name: "+prb_pair_name+", "+title_form)
        ax.scatter(pd.to_datetime(t,unit='s'),T)
        ax.set_ylim([Theta_low_bnd,Theta_upp_bnd])
        ax.set_ylabel('|Theta| (degrees)')   
        ax.xaxis.set_major_formatter(mpl.dates.DateFormatter(ax_form))
        ax.xaxis.set_minor_formatter(mpl.dates.DateFormatter(ax_form))
    
        ax.legend([prb_pair_name])
        
        print("Done!\n")
        return fig, ax

    def plot_z_ratio_vs_time(self, prb_pair = None, prb_pair_name = None, df = pd.DataFrame(), f_l = None, f_h = None, ratio_l_bnd = None, ratio_h_bnd = None):
        """
            plot_z_ratio_vs_time plots the ratio z_l/z_h, impedance at a low frequency over impedance at a high frequency, for a particular probe pair or name. 
            
            Inputs
            ------
            (ONE OF THE FOLLOWING IS REQUIRED)
            prb_pair: string. 
                The probe pair of the data to plot.  
            prb_pair_name : string. 
                The probe pair name of the data to plot.
                
            df : (optional) pandas dataframe. 
                Data to be plotted. Should have the same format as all other Multi-PIP data. Defaults to self.data.
            f_l : (optional) integer . 
                The frequency value to use when determining low impedance. Default is minimum frequency found in the data.
            f_h : (optional) integer. 
                The frequency value to use when determining high impedance. Default is maxmimum frequency found in the data.
            ratio_l_bnd : (optional) float.
                The lower bound for the simple ratio. Default is 'ratio_l_bnd = simple_r.min() - 10'.
            ratio_h_bnd : (optional) float.
                The lower bound for the simple ratio. Default is 'ratio_h_bnd = simple_r.max() + 10'.
                
            Outputs
            -------
            fig : matplotlib figure.
                Figure associated with impedance ratio versus time.
            ax : matplotlib axes.
                Axis associated with impedance ratio versus time.
                
                
        """
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
            
        # Find frequency vector in data
        freq = pd.unique(df['frequency'])
        # Check if frequencies inputted are in vector
        if not (f_l in freq) & (f_h in freq):
            f_l = freq.min()
            f_h = freq.max()
            print("Could not find frequencies inputted in data, defaulting to :\nf_l = "+str(f_l)+", f_h = "+str(f_h))
            
        
        #Check wether inputs correspond to good data
        prb_pair, prb_pair_name = self.prb_pair_name_crosscheck(df, prb_pair, prb_pair_name)
        
        print("Plotting low/high impedance ratio at "+str(f_l)+", "+str(f_h)+" Hz for probe pair name "+prb_pair_name+" and associated probe pair "+prb_pair+"...")
        

        # Find indices associated with high and low frequency and probe pair
        ind_f_l = df.loc[(df['frequency'] == f_l) & (df['probe pair name'] == prb_pair_name)]
        ind_f_h = df.loc[(df['frequency'] == f_h) & (df['probe pair name'] == prb_pair_name)]
                    
        # Get impedance and time
        z_l = ind_f_l['impedance']
        z_h = ind_f_h['impedance']
        
        t = ind_f_l['Time']
        
        # Get initial and final time
        t_i = pd.to_datetime(t.min(),unit='s')
        t_f = pd.to_datetime(t.max(),unit='s')
        
        # Get datetime title and axis format
        title_form, ax_form = self.format_datetime_plot(t_i,t_f)
        
        # Calculate simple ratio
        simple_r = np.divide(np.asarray(z_l),np.asarray(z_h))
        #print(simple_r)
        # Set axis bounds for high and low impedance
        if ratio_h_bnd == None:
            ratio_h_bnd = simple_r.max()+10
        
        if ratio_l_bnd == None:
            ratio_l_bnd = simple_r.min()-10
        
        
        # Plot high versus low impedance
        fig, ax = plt.subplots(1, figsize = (12,8))
        fig.suptitle("Simple ratio of impedances at ("+str(f_l)+","+str(f_h)+" Hz), Probe: "+prb_pair+", Name: "+prb_pair_name+", "+title_form)
        ax.scatter(pd.to_datetime(t,unit='s'),simple_r)
        ax.set_ylim([ratio_l_bnd,ratio_h_bnd])
        ax.set_ylabel('Z_l/Z_h')
        ax.xaxis.set_major_formatter(mpl.dates.DateFormatter(ax_form))
        ax.xaxis.set_minor_formatter(mpl.dates.DateFormatter(ax_form))
                    
        print(" Done!\n\n")
        return fig, ax
    
    def plot_R_vs_I(self, prb_pair = None, prb_pair_name = None, df = pd.DataFrame):
        """
            plot_R_vs_I takes care of all of the formatting for creating a real versus imaginary plot. 
            
            Inputs
            ------
            (ONE OF THE FOLLOWING IS REQUIRED)
            prb_pair: string. 
                The probe pair of the data to plot.  
            prb_pair_name : string. 
                The probe pair name of the data to plot. 
                
            df : (optional) pandas dataframe. 
                Data to be plotted. Should have the same format as all other Multi-PIP data. Defaults to self.data.
                
            Outputs
            -------
            fig : matplotlib figure.
                Figure associated with the real versus imaginary plot.
            ax : matplotlib axes.
                Axis associated with the real versus imaginary plot.
            
        """
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
        
        #Check wether inputs correspond to good data
        prb_pair, prb_pair_name = self.prb_pair_name_crosscheck(df, prb_pair, prb_pair_name)
        
        print("Plotting real versus imaginary for probe pair name "+prb_pair_name+" and associated probe pair "+prb_pair+"...")
            
    
        # Find indices associated with frequency and probe pair
        ind = df.loc[(df['probe pair name'] == prb_pair_name)]
        
                    
        # Get resistance and reactance
        real = ind['real']
        imag = ind['imaginary']
        
        # Plot impedance
        fig, ax = plt.subplots(1, figsize = (12,8))
        fig.suptitle("Real versus Imaginary, Probe: "+prb_pair+", Name: "+prb_pair_name)
        ax.scatter(real,imag)
        ax.set_ylabel('Real')   
        ax.set_xlabel('Imaginary')
        ax.legend([prb_pair_name])
        
        print("Done!\n")
        return fig, ax
    
    def plot_z_h_vs_T_h(self, freq, prb_pair = None, prb_pair_name = None, df = pd.DataFrame(), z_upp_bnd = None, z_low_bnd = None, Theta_upp_bnd = None, Theta_low_bnd = None):
        """
            plot_z_h_vs_T_h takes care of all of the formatting for creating a 
            impedance vs |phase angle| plot for a particular frequency. 
            
            Inputs
            ------
            freq : (required) integer.
                The frequency for which impedance is to be plotted. 
                
            (ONE OF THE FOLLOWING IS REQUIRED)
            prb_pair: string. 
                The probe pair of the data to plot.  
            prb_pair_name : string. 
                The probe pair name of the data to plot. 
            
             df : (optional) pandas dataframe. 
                Data to be plotted. Should have the same format as all other Multi-PIP data. Defaults to self.data.
            z_low_bnd : (optional) float.
                The lower bound for impdance in kOhms. Default is 'z_low_bnd = z.min() - 10 (kOhms)'.
            z_upp_bnd : (optional) float.
                The upper bound for impdance in kOhms. Default is 'z_upp_bnd = z.max() + 10 (kOhms)'.
            Theta_upp_bnd : (optional) float.
                The upper bound for the phase angle vs time plot in degrees. Default is 'Theta_upp_bnd = T.max() + 5'.
            Theta_low_bnd : (optional) float.
                The lower bound for the phase angle vs time plot in degrees. Default is 'Theta_low_bnd = T.min() - 5'.
                
            Outputs
            -------
            fig : matplotlib figure.
                Figure associated with the impedance versus phase plot.
            ax : matplotlib axes.
                Axis associated with the impedance versus phase plot.
            
        """
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
        
        #Check that given frequency is part of the data set
        frequencies = pd.unique(df['frequency'])
        if not freq in frequencies:
            raise TypeError("The frequency you inputted does not match any of the frequencies in the data.")
        
        #Check wether inputs correspond to good data
        prb_pair, prb_pair_name = self.prb_pair_name_crosscheck(df, prb_pair, prb_pair_name)
        
        print("Plotting impedance versus |phase angle| at "+str(freq)+" Hz for probe pair name "+prb_pair_name+" and associated probe pair "+prb_pair+"...")
            
    
        # Find indices associated with frequency and probe pair
        ind = df.loc[(df['frequency'] == freq) & (df['probe pair name'] == prb_pair_name)]
        
                    
        # Get |phase| angle and time
        T = ind['phase'].abs()
        z = ind['impedance']/1000
        
        # Set upper and lower axis bounds for impedance
        if Theta_upp_bnd == None:
            Theta_upp_bnd = T.max() + 5
            
        if Theta_low_bnd == None:
            Theta_low_bnd = T.min() - 5
        
        if z_upp_bnd == None:
            z_upp_bnd = z.max() + 10
            
        if z_low_bnd == None:
            z_low_bnd = z.min() - 10
        
        # Plot impedance
        fig, ax = plt.subplots(1, figsize = (12,8))
        fig.suptitle("impedance vs |Phase angle| at "+str(freq)+" Hz, Probe: "+prb_pair+", Name: "+prb_pair_name)
        ax.scatter(T,z)
        ax.set_xlim([Theta_low_bnd,Theta_upp_bnd])
        ax.set_ylim([z_low_bnd,z_upp_bnd])
        ax.set_ylabel('Z (kOhms)') 
        ax.set_xlabel('|Phase angle| (degrees)')
    
        ax.legend([prb_pair_name])
        
        print("Done!\n")
        return fig, ax
    
    def save_NDZI(self, path = None):
        """
            save_NDZI saves NDZI to a CSV file with the metadata of the imported data file as a header. 
            By default, the CSV file is saved as 'NDZI_(low frequency)_(high frequency)_Hz_(name).CSV' to the current working directory. 
            
            Columns included in save are : Time, probe pair, probe pair name, NDZI, high frequency, and low frequency.
            
            Inputs
            ------
            path : (optional) string.
                The path and filename to save NDZI to. 
        """
        # Check if NDZI is computed. If it is not, compute it with default settings. 
        if self.NDZI.empty:
            print("No NDZI found, computing default NDZI for highest and lowest frequencies found in data. \n")
            freq = pd.unique(self.data['frequency'])
            if freq.size == 0:
                raise Exception("No data found, please import a CSV file.")
            f_l = freq[0]
            f_h = freq[-1]
            self.calc_NDZI(f_l, f_h)
            
        # Get frequencies of NDZI 
        f_h = pd.unique(self.NDZI['High freq'])
        f_l = pd.unique(self.NDZI['Low freq'])
            
        if path == None:
            print("No path specified, saving to ./\n")
            path = './NDZI_'+str(f_l[0])+'_'+str(f_h[0])+'_Hz_'+self.name+'.CSV'
        
        
        print("Saving NDZI to: "+path)
        
        # Open and erase data already present in file
        f = open(path,'w')
        f.truncate()
        f.close()
        
        f = open(path,'a',newline='')
        
        # Write metadata and data to csv file.
        self.metadata.to_csv(f, line_terminator="", index = False)
        f.write("\n")
        self.NDZI.to_csv(f, line_terminator="", index = False)
        
        f.close()
        print("Done!")
        
    def save_Phase(self, path = None, df = pd.DataFrame()):
        """
            save_Phase saves the phase angle to a CSV file with the metadata of the imported data file as a header. 
            By default, the CSV file is saved as 'Phase_(name).CSV' to the current working directory. 
            
            Columns included in save are : Time, probe pair, probe pair name, frequency, and phase angle.
            
            Inputs
            ------
            path : (optional) string.
                The path and filename to save phase angle to. 
            df : (optional) pandas dataframe. 
                Data from which to save. Should have the same format as all other Multi-PIP data. Defaults to self.data.
                
        """
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
            # Check if data is present. If it is not, throw an exception. 
            if self.data.empty:
                raise Exception("No data found, please import a data file!")

        
            
        # Find phase data
        p_data = df[['Time','probe pair', 'probe pair name', 'frequency', 'phase']]
        
        if path == None:
            print("No path specified, saving to working directory!\n")
            path = './Phase_'+self.name+'.CSV'
        
        
        print("Saving phase to: "+path)
        
        # Open and erase data already present in file
        f = open(path,'w')
        f.truncate()
        f.close()
        
        f = open(path,'a',newline='')
        
        # Write metadata and data to csv file.
        self.metadata.to_csv(f, line_terminator="", index = False)
        f.write("\n")
        p_data.to_csv(f, line_terminator="", index = False)
        
        f.close()
        print("Done!")
        
    def save_Real_Imaginary(self, path = None, df = pd.DataFrame()):
        """
            save_Real_Imaginary saves the real and imaginary data to a CSV file with the metadata of the imported data file as a header. 
            By default, the CSV file is saved as 'Real_Imaginary_(name).CSV' to the current working directory. 
            
            Columns included in save are : Time, probe pair, probe pair name, frequency, real, and imaginary.
            
            Inputs
            ------
            path : (optional) string.
                The path and filename to save phase angle to. 
            df : (optional) pandas dataframe. 
                Data from which to save. Should have the same format as all other Multi-PIP data. Defaults to self.data.
        """
        
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
            # Check if data is present. If it is not, throw an exception. 
            if self.data.empty:
                raise Exception("No data found, please import a data file!")
            
        # Find data and rename real and imaginary to resistance and reactance
        R_data = df[['Time','probe pair', 'probe pair name', 'frequency', 'real','imaginary']]
        
        if path == None:
            print("No path specified, saving to working directory!\n")
            path = './Real_Imaginary_'+self.name+'.CSV'
        
        
        print("Saving real and imaginary data to: "+path)
        
        # Open and erase data already present in file
        f = open(path,'w')
        f.truncate()
        f.close()
        
        f = open(path,'a',newline='')
        
        # Write metadata and data to csv file.
        self.metadata.to_csv(f, line_terminator="", index = False)
        f.write("\n")
        R_data.to_csv(f, line_terminator="", index = False)
        
        f.close()
        print("Done!")
        
    def save_Impedance(self, path = None, df = pd.DataFrame()):
        """
            save_Impedance saves the impedance data to a CSV file with the metadata of the imported data file as a header. 
            By default, the CSV file is saved as 'Impedance_(name).CSV' to the current working directory. 
            
            Columns included in save are : Time, probe pair, probe pair name, frequency, and impedance.
            
            Inputs
            ------
            path : (optional) string.
                The path and filename to save phase angle to.
            df : (optional) pandas dataframe. 
                Data from which to save. Should have the same format as all other Multi-PIP data. Defaults to self.data.
        """
        
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
            # Check if data is present. If it is not, throw an exception. 
            if self.data.empty:
                raise Exception("No data found, please import a data file!")
            
        # Find impedance data
        Z_data = df[['Time','probe pair', 'probe pair name', 'frequency', 'impedance']]
        
        if path == None:
            print("No path specified, saving to working directory!\n")
            path = './Impedance_'+self.name+'.CSV'
        
        
        print("Saving impedance to: "+path)
        
        # Open and erase data already present in file
        f = open(path,'w')
        f.truncate()
        f.close()
        
        f = open(path,'a',newline='')
        
        # Write metadata and data to csv file.
        self.metadata.to_csv(f, line_terminator="", index = False)
        f.write("\n")
        Z_data.to_csv(f, line_terminator="", index = False)
        
        f.close()
        print("Done!")
        
    def save_Temperature(self, path = None, df = pd.DataFrame()):
        """
            save_Temperature saves the temperature data to a CSV file with the metadata of the imported data file as a header. 
            By default, the CSV file is saved as 'Temperature_(name).CSV' to the current working directory. 
            
            Columns included in save are : Time, probe pair, probe pair name, frequency, and temperature.
            
            Inputs
            ------
            path : (optional) string.
                The path and filename to save phase angle to. 
            df : (optional) pandas dataframe. 
                Data from which to save. Should have the same format as all other Multi-PIP data. Defaults to self.data.
        """
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
            # Check if data is present. If it is not, throw an exception. 
            if self.data.empty:
                raise Exception("No data found, please import a data file!")
            
        # Find temperature data
        T_data = df[['Time','probe pair', 'probe pair name', 'frequency', 'temperature']]
        
        if path == None:
            print("No path specified, saving to working directory!\n")
            path = './Temperature_'+self.name+'.CSV'
        
        
        print("Saving temperature to: "+path)
        
        # Open and erase data already present in file
        f = open(path,'w')
        f.truncate()
        f.close()
        
        f = open(path,'a',newline='')
        
        # Write metadata and data to csv file.
        self.metadata.to_csv(f, line_terminator="", index = False)
        f.write("\n")
        T_data.to_csv(f, line_terminator="", index = False)
        
        f.close()
        print("Done!")
        
    def save_Magnitude(self, path = None, df = pd.DataFrame()):
        """
            save_Magnitude saves the magnitude data to a CSV file with the metadata of the imported data file as a header. 
            By default, the CSV file is saved as 'Magnitude_(name).CSV' to the current working directory. 
            
            Columns included in save are : Time, probe pair, probe pair name, frequency, and magnitude.
            
            Inputs
            ------
            path : (optional) string.
                The path and filename to save phase angle to. 
            df : (optional) pandas dataframe. 
                Data to be saved. Should have the same format as all other Multi-PIP data. Defaults to self.data.
        """
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
            # Check if data is present. If it is not, throw an exception. 
            if self.data.empty:
                raise Exception("No data found, please import a data file!")
            
        # Find temperature data
        M_data = df[['Time','probe pair', 'probe pair name', 'frequency', 'magnitude']]
        
        if path == None:
            print("No path specified, saving to working directory!\n")
            path = './Magnitude_'+self.name+'.CSV'
        
        
        print("Saving Magnitude to: "+path)
        
        # Open and erase data already present in file
        f = open(path,'w')
        f.truncate()
        f.close()
        
        f = open(path,'a',newline='')
        
        # Write metadata and data to csv file.
        self.metadata.to_csv(f, line_terminator="", index = False)
        f.write("\n")
        M_data.to_csv(f, line_terminator="", index = False)
        
        f.close()
        print("Done!")
        
    def z_window(self, z_l, z_h, df = pd.DataFrame()):
        """
            z_window is a data filtering function for impedance. It finds all data associated with the lowest frequency impedance
            data that falls within a user-inputted impedance window.
            It then removes that data point along with the associated frequency measurements.
            For example, if low_pass finds that a measurement at 10:30 PM at 100 Hz has exceeded the threshold, it will
            remove all other frequency data (e.g. 1000 Hz, 6000 Hz,..., 95000 Hz) for 10:30 PM.
            
            Note: 
                z_window chooses the frequencies to remove based on the following N = (number of frequencies) rows after 
                the low frequency point that has exceeded the impedance threshold. 
                As a result, any formatting changes to the Multi-PIP data will most likely have an effect on the performance
                of z_window.
            
            Error note:
                If you are getting errors related to NDZI computation or plotting, check to make sure there are no 
                redundencies in your frequency data. 
            
            Inputs
            ------
            z_l : (required) number.
                Impdedance window lower limit value in kOhms, above which data is to be removed. 
            z_h : (required) number.
                Impdedance window upper limit value in kOhms, above which data is to be removed. 
            df : (optional) pandas dataframe. 
                Data to be filtered. Should have the same format as all other Multi-PIP data. Defaults to self.data.
                
            Outputs 
            -------
            filt : pandas dataframe.
                The filtered data.
        """
        
        if df.empty:
            # If no dataframe is passed in, then default to data stored in self.
            df = self.data
            # Check if data is present. If it is not, throw an exception. 
            if self.data.empty:
                raise Exception("No data found, please import a data file!")
        
        # Find frequencies present in data and select minimum
        f_vec = pd.unique(df['frequency'])
        # Debugging line
        #print("Freq " + str(len(f_vec)))
        freq = f_vec.min()
            
        print("Setting impedance window at ["+str(z_l)+"-"+str(z_h)+"] kOhms for frequency "+ str(freq)+" Hz...")
        # Find the row numbers to remove for particular frequency
        f_rmv = df.loc[(df['frequency'] == freq) & ((df['impedance']/1000 > z_h) | (df['impedance']/1000 < z_l))]
        print("Found "+str(len(f_rmv.index))+" data points to remove for "+str(freq)+" Hz. \nRemoving now...")
        
        # Filter the rest of the associated data out based on number of frequencies
        filt = df
        for i in f_rmv.index:
            # Drop the next N (= number of frequencies) rows.
            drp = np.linspace(i,(i+len(f_vec)-1), len(f_vec), dtype=int)
            filt = filt.drop(index = drp)
        
        #Check that the correct number of points have been filtered out
        #if len(df.index) - len(filt.index) != len(f_rmv.index)*len(f_vec):
            #raise Exception("Something went wrong filtering... ")
        
        print('Done! Removed a total of '+str(len(df.index) - len(filt.index))+ ' data points.')
        return filt
        