import numpy as np
def second_derivative(data,t):
	"""
		This function calculates the second derivative of a data trace in time.

		Last Modified: 2/12/2021

		Inputs
		------
		data: numpy array
			data trace from which the second derivative is to be calculated
		t: numpy array
			time array from for second derivative calculation (defined UNIX time)

		Outputs
		-------
		d2t: numpy array
			the second derivative of your data
		t_d2t: numpy array 
			the timeseries associated with d2t

		NOTE: 
		This derivative is calculated using the second order forward difference method. 
		As a result, the second derivative and associated time array are two entries shorter than the input arrays.
		In essence, the finite second derivative of the trace "exists" at the earliest timestamp used to calculate the derivative
		.
		For more information, please see: https://en.wikipedia.org/wiki/Finite_difference


	"""

	# Ensuring arrays have the same dimension and are of appropriate type
	if not isinstance(data,np.ndarray) & isinstance(t,np.ndarray):
		raise TypeError("Input arrays are not of type numpy.ndarray")

	if not data.size == t.size:
		raise Exception("Input arrays are not the same dimension.")

	# Calculate second derivative and associated time series
	d2t = np.zeros(data.size - 2)
	t_d2t = np.zeros(t.size - 2)

	for i in range(len(data)-2):
		t_d2t[i] = t[i]
		d2t[i] = (data[i+2]-2*data[i+1]+data[i])/((t[i+1]-t[i])**2)

	return d2t, t_d2t


