import numpy as np
def derivative(data,t):
	"""
		This function calculates the derivative of a data trace in time.

		Last Modified: 2/12/2021

		Inputs
		------
		data: numpy array
			data trace from which the derivative is to be calculated
		t: numpy array
			time array from for derivative calculation (defined UNIX time)

		Outputs
		-------
		dt: numpy array
			the derivative of your data
		t_dt: numpy array 
			the timeseries associated with dt

		NOTE: 
		This derivative is calculated using the finite forward difference quotient. 
		As a result, the derivative and associated time array are one entry shorter than the input arrays.
		In essence, the finite derivative of the trace "exists" at the earlier timestamp used to calculate the derivative
		.
		For more information, please see: https://en.wikipedia.org/wiki/Finite_difference


	"""

	# Ensuring arrays have the same dimension and are of appropriate type
	if not isinstance(data,np.ndarray) & isinstance(t,np.ndarray):
		raise TypeError("Input arrays are not of type numpy.ndarray")

	if not data.size == t.size:
		raise Exception("Input arrays are not the same dimension.")

	# Calculate derivative and associated time series
	dt = np.zeros(data.size - 1)
	t_dt = np.zeros(t.size - 1)

	for i in range(len(data)-1):
		t_dt[i] = t[i]
		dt[i] = (data[i+1]+data[i])/(t[i+1]-t[i])

	return dt, t_dt


